"""
===============================================================================
    Projekt: Moduláris Esőkert Számítási Programok
    Fejlesztő: Glia Kutatás - Fejlesztési Innovációs Kft.
    Weboldal: www.glia.hu
    Verzió: 1.0
    Dátum: 2018
    Licenc: Nyílt forráskódú szoftver (MIT Licenc)

    Pályázati támogatás:
    Ez a fejlesztés a GINOP-2.1.7-15-2016-02435 pályázat keretén belül 
    valósult meg, amely célja a moduláris beton-műanyag kompozit szikkasztó 
    rendszer fejlesztése volt és a Glia Kft. tarnabodi telephelyén került megvalósításra.

    Leírás:
    Ez a program a moduláris esőkert különböző hidrológiai paramétereinek 
    számítására szolgál. Tartalmazza az alábbi funkciókat:
    - Kapilláris vízemelés számítása
    - Infiltráció és vízbeszivárgás számítása
    - Vízmozgás sebességének becslése
    - Párolgás és evapotranszspiráció számítása
    - Esőkert tározási kapacitás és túlcsordulás vizsgálata

    Licencfeltételek:
    Ez a szoftver az MIT Licenc alatt áll, így szabadon másolható, módosítható 
    és terjeszthető, feltéve hogy a fent megnevezett fejlesztői jogokat 
    és a licencfeltételeket tiszteletben tartják.
===============================================================================
"""


import argparse

def penman_monteith_et(rn, g, u2, es_ea, gamma, delta, t):
    """
    Kiszámítja a napi evapotranszspirációt (ET) a Penman-Monteith egyenlet alapján.

    Paraméterek:
        rn (float): Nettó sugárzás (MJ/m²/nap)
        g (float): Talajhőmérséklet változása (MJ/m²/nap)
        u2 (float): Szélsebesség 2 m magasságban (m/s)
        es_ea (float): Telítési és aktuális páranyomás különbsége (kPa)
        gamma (float): Pszichrometrikus állandó (kPa/°C)
        delta (float): Telítési görbe meredeksége (kPa/°C)
        t (float): Középhőmérséklet (°C)

    Visszatérés:
        float: Napi referencia evapotranszspiráció (ET) mm/nap
    """
    et0 = (0.408 * delta * (rn - g) + gamma * (900 / (t + 273)) * u2 * es_ea) / (delta + gamma * (1 + 0.34 * u2))
    return et0

def main():
    """
    A program fő része, amely parancssori argumentumokat dolgoz fel,
    és kiszámolja az evapotranszspiráció értékét.
    """
    parser = argparse.ArgumentParser(description="Párolgás és Evapotranszspiráció számítása (Penman-Monteith).")

    # Parancssori argumentumok meghatározása
    parser.add_argument("--rn", type=float, required=True, help="Nettó sugárzás (MJ/m²/nap)")
    parser.add_argument("--g", type=float, required=True, help="Talajhőmérséklet változása (MJ/m²/nap)")
    parser.add_argument("--u2", type=float, required=True, help="Szélsebesség 2m magasságban (m/s)")
    parser.add_argument("--es_ea", type=float, required=True, help="Telítési és aktuális páranyomás különbsége (kPa)")
    parser.add_argument("--gamma", type=float, required=True, help="Pszichrometrikus állandó (kPa/°C)")
    parser.add_argument("--delta", type=float, required=True, help="Telítési görbe meredeksége (kPa/°C)")
    parser.add_argument("--t", type=float, required=True, help="Középhőmérséklet (°C)")

    # Argumentumok beolvasása
    args = parser.parse_args()

    # Számítás végrehajtása
    et_value = penman_monteith_et(args.rn, args.g, args.u2, args.es_ea, args.gamma, args.delta, args.t)

    # Eredmény kiírása
    print("\n🌞 Párolgás és Evapotranszspiráció számítás:")
    print(f"💧 Napi referencia evapotranszspiráció (ET): {et_value:.2f} mm/nap")

if __name__ == "__main__":
    main()
