<!DOCTYPE html>
<html lang="hu">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Programkód Megjelenítése</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #f4f4f4;
            padding: 20px;
        }
        .code-container {
            background: #282c34;
            color: #abb2bf;
            padding: 15px;
            border-radius: 5px;
            font-family: "Courier New", monospace;
            overflow-x: auto;
            white-space: pre;
        }
    </style>
</head>
<body>

    <h2>Kapilláris vízemelés számítása Python Kód</h2>
    <div class="code-container">
        <pre><code>

"""
===============================================================================
    Projekt: Moduláris Esőkert Számítási Programok
    Fejlesztő: Glia Kutatás - Fejlesztési Innovációs Kft.
    Weboldal: www.glia.hu
    Verzió: 1.0
    Dátum: 2018
    Licenc: Nyílt forráskódú szoftver (MIT Licenc)

    Pályázati támogatás:
    Ez a fejlesztés a GINOP-2.1.7-15-2016-02435 pályázat keretén belül 
    valósult meg, amely célja a moduláris beton-műanyag kompozit szikkasztó 
    rendszer fejlesztése volt és a Glia Kft. tarnabodi telephelyén került megvalósításra.

    Leírás:
    Ez a program a moduláris esőkert különböző hidrológiai paramétereinek 
    számítására szolgál. Tartalmazza az alábbi funkciókat:
    - Kapilláris vízemelés számítása
    - Infiltráció és vízbeszivárgás számítása
    - Vízmozgás sebességének becslése
    - Párolgás és evapotranszspiráció számítása
    - Esőkert tározási kapacitás és túlcsordulás vizsgálata

    Licencfeltételek:
    Ez a szoftver az MIT Licenc alatt áll, így szabadon másolható, módosítható 
    és terjeszthető, feltéve hogy a fent megnevezett fejlesztői jogokat 
    és a licencfeltételeket tiszteletben tartják.
===============================================================================
"""


import argparse
import math

def horton_infiltration(f0, fc, k, time):
    """
    Horton-féle infiltrációs egyenlet alapján kiszámolja az adott időre vonatkozó
    beszivárgási sebességet és az összes beszivárgott vízmennyiséget.

    Paraméterek:
        f0 (float): Kezdeti infiltrációs sebesség (mm/h)
        fc (float): Állandósult infiltrációs sebesség (mm/h)
        k (float): Infiltrációs csökkenési állandó (1/h)
        time (float): Idő (óra)

    Visszatérés:
        dict: Infiltrációs sebesség (mm/h) és teljes beszivárgott vízmennyiség (mm)
    """
    # Infiltrációs sebesség adott időpillanatban
    ft = fc + (f0 - fc) * math.exp(-k * time)

    # Teljes infiltrált vízmennyiség adott idő alatt
    Ft = fc * time + ((f0 - fc) / k) * (1 - math.exp(-k * time))

    return {
        "Aktuális infiltrációs sebesség (f_t)": ft,
        "Összes beszivárgott vízmennyiség (F_t)": Ft
    }

def main():
    """
    Parancssori argumentumokat dolgoz fel, és kiszámolja az infiltráció sebességét és a vízbeszivárgást.
    """
    parser = argparse.ArgumentParser(description="Horton-féle infiltrációs sebesség és vízbeszivárgás számítása.")

    # Parancssori argumentumok meghatározása
    parser.add_argument("--initial_infiltration", type=float, required=True, help="Kezdeti infiltrációs sebesség f0 (mm/h)")
    parser.add_argument("--final_infiltration", type=float, required=True, help="Állandósult infiltrációs sebesség fc (mm/h)")
    parser.add_argument("--decay_rate", type=float, required=True, help="Infiltrációs csökkenési állandó k (1/h)")
    parser.add_argument("--time", type=float, required=True, help="Idő t (óra)")

    # Argumentumok beolvasása
    args = parser.parse_args()

    # Számítások végrehajtása
    results = horton_infiltration(args.initial_infiltration, args.final_infiltration, args.decay_rate, args.time)

    # Eredmények kiírása
    print("\n📊 Infiltráció és Vízbeszivárgás számítás:")
    print(f"🌧 Aktuális infiltrációs sebesség (f_t): {results['Aktuális infiltrációs sebesség (f_t)']:.2f} mm/h")
    print(f"💦 Összes beszivárgott vízmennyiség (F_t): {results['Összes beszivárgott vízmennyiség (F_t)']:.2f} mm")

if __name__ == "__main__":
    main()
    
    
      </code></pre>
    </div>

</body>
</html>
